# MiBa Video Manager - YouTube API Services Implementation Document

**Application Name:** MiBa Video Manager  
**Developer:** Michael Bachmann  
**Project ID:** miba-video-manager  
**Date:** 2025-01-XX  
**YouTube API Version:** v3

---

## Executive Summary

MiBa Video Manager is a desktop application built with Electron that enables content creators to manage local video files and seamlessly upload them to YouTube. The application integrates with YouTube Data API v3 to provide video upload, metadata management, and library synchronization functionality. This document details the implementation, architecture, security measures, and usage patterns of YouTube API Services within the application.

**Demo Video:** https://www.youtube.com/watch?v=3gCD1y3SQhs  
**Application Website:** https://www.miba.dk/miba-video-manager

---

## 1. Application Overview

### 1.1 Purpose and Target Audience

MiBa Video Manager serves as a creator tool designed for individual content creators and small video producers who need an efficient desktop solution for YouTube content management. The application bridges the gap between local video file management and YouTube publishing, allowing users to:

- Organize and browse local video files
- Upload videos directly to YouTube with custom metadata
- Update video metadata (title, description, privacy settings) on existing YouTube videos
- Manage and synchronize YouTube video library
- Delete videos from YouTube when needed

### 1.2 Technical Stack

- **Framework:** Electron (desktop application)
- **Runtime:** Node.js
- **YouTube API Client:** googleapis v131.0.0
- **YouTube API Version:** v3
- **Authentication:** Google OAuth 2.0 Desktop Flow with PKCE
- **Supported Platforms:** Windows (x64), macOS (x64, arm64)

---

## 2. Architecture and Design

### 2.1 Application Structure

The application follows a traditional Electron architecture:

```
┌─────────────────────────────────────┐
│     Renderer Process (UI)           │
│  - User interface and interactions  │
│  - Video gallery and player         │
└──────────────┬──────────────────────┘
               │ IPC Communication
┌──────────────▼──────────────────────┐
│     Main Process (Node.js)          │
│  - YouTube API integration          │
│  - OAuth authentication             │
│  - File system operations           │
│  - Data persistence                 │
└──────────────┬──────────────────────┘
               │
┌──────────────▼──────────────────────┐
│     YouTube Data API v3             │
│  - Video uploads                    │
│  - Metadata operations              │
│  - Library retrieval                │
└─────────────────────────────────────┘
```

### 2.2 Core Components

**YouTube Service (`src/main/youtube.js`)**
- Main service class handling all YouTube API interactions
- Manages authentication, API calls, caching, and data persistence
- Implements upload jobs with progress tracking
- Handles multi-account support

**Google OAuth Module (`src/main/auth/googleOAuth.js`)**
- Implements secure OAuth 2.0 desktop flow with PKCE
- Manages token storage and refresh
- Handles multiple account authentication
- Implements CSRF protection

**Video Store (`src/main/auth/youtubeAccountStore.js`)**
- Persists video records linking local files to YouTube videos
- Stores per-account data in local JSON files
- Maintains upload status and metadata

---

## 3. YouTube API Integration

### 3.1 API Services Used

The application exclusively uses **YouTube Data API v3**:

| API Endpoint | Purpose | Quota Cost |
|-------------|---------|------------|
| `channels.list` | Retrieve channel information and uploads playlist ID | ~1 unit |
| `videos.insert` | Upload video files to YouTube | 1,600 units |
| `videos.update` | Update video metadata (title, description, privacy, etc.) | ~50 units |
| `videos.delete` | Delete videos from YouTube | ~50 units |
| `videos.list` | Retrieve detailed video information | ~1-2 units |
| `playlistItems.list` | Fetch videos from uploads playlist | ~1 unit |
| `playlists.list` | Retrieve playlist information | ~1 unit |

### 3.2 Authentication Implementation

#### 3.2.1 OAuth 2.0 Desktop Flow with PKCE

The application implements Google's recommended secure OAuth 2.0 desktop flow using Proof Key for Code Exchange (PKCE) to meet Google's security requirements.

**Security Controls:**

1. **PKCE Implementation:**
   - Each authentication attempt generates a unique per-session `code_verifier` (cryptographically random)
   - Derives a `code_challenge` using S256 (SHA256) method
   - The verifier is supplied when exchanging authorization codes, binding tokens to the originating client instance

2. **CSRF Protection:**
   - High-entropy, URL-safe `state` token attached to authorization requests
   - Local callback handler verifies returned `state` matches session token before code exchange
   - Prevents cross-site request forgery attacks

3. **Ephemeral Session Secrets:**
   - PKCE verifiers and `state` tokens stored entirely in memory
   - Never written to disk
   - Cleared immediately after successful token exchange, cancellation, timeout, or callback errors

4. **Local Redirect Binding:**
   - Embedded HTTP callback server accepts requests only on `127.0.0.1:4587`
   - Server shuts down immediately after handling response
   - Reduces window for spoofed callbacks

**OAuth Scopes:**
- `youtube` - Basic YouTube API access
- `youtube.upload` - Upload videos to YouTube
- `youtube.force-ssl` - Ensure secure connections

**Token Storage:**
- OAuth tokens stored securely using Electron's secure token storage
- Tokens automatically refreshed when expired
- Per-account token management for multi-account support

#### 3.2.2 Authentication Flow

```
1. User initiates authentication from Settings UI
2. Application generates PKCE code_verifier and code_challenge
3. Application generates CSRF state token
4. Application opens browser with OAuth authorization URL
5. User authenticates with Google in browser
6. Google redirects to local callback server (127.0.0.1:4587)
7. Application verifies state token matches session
8. Application exchanges authorization code using code_verifier
9. Google returns OAuth tokens
10. Tokens stored securely in Electron secure storage
11. Application uses tokens for YouTube API calls
```

**Reference:** See `docs/auth/google-oauth.md` for detailed security documentation.

### 3.3 API Usage Patterns

#### 3.3.1 Video Upload (`videos.insert`)

**Implementation Location:** `src/main/youtube.js` - `startUpload()` method

**Process:**
1. Validates video file exists and is readable
2. Authenticates user and retrieves YouTube API client
3. Validates user account ID
4. Creates upload job with unique ID for progress tracking
5. Prepares video metadata:
   - Title (from metadata or filename, max 100 characters)
   - Description (from metadata or empty)
   - Privacy status (public, private, or unlisted)
   - Category ID (default: 22 - People & Blogs)
   - Self-declared made for kids: false
6. Streams video file to YouTube API with progress tracking
7. Emits progress updates to UI (every 1% increment)
8. On completion, saves video record linking local file to YouTube video ID
9. Invalidates library cache to trigger refresh

**Key Features:**
- Real-time upload progress tracking
- Supports cancellation (job cleanup)
- Error handling with user-friendly messages
- Automatic retry handling (handled by googleapis library)
- Prevents duplicate uploads (checks for active jobs)

**Code Example:**
```javascript
const response = await youtube.videos.insert({
  part: ['snippet', 'status'],
  requestBody: {
    snippet: {
      title: title.slice(0, 100),
      description: description,
      categoryId: '22'
    },
    status: {
      privacyStatus: privacy, // 'public', 'private', or 'unlisted'
      selfDeclaredMadeForKids: false
    }
  },
  media: {
    body: fileStream // Readable stream of video file
  }
});
```

#### 3.3.2 Metadata Update (`videos.update`)

**Implementation Location:** `src/main/youtube.js` - `updateMetadata()` method

**Process:**
1. Authenticates user and retrieves YouTube API client
2. Retrieves existing video details from YouTube API
3. Merges user-provided metadata with existing data
4. Preserves settings not being updated (e.g., embeddable, tags)
5. Updates video with new metadata
6. Updates local video record
7. Invalidates library cache

**Supported Metadata:**
- Title (max 100 characters)
- Description
- Privacy status (public, private, unlisted)
- Category ID
- Tags (array of strings)
- Default language
- Default audio language
- Embeddable status (preserved if not specified)

**Key Features:**
- Preserves existing metadata when not updating
- Validates privacy status values
- Maintains embeddable status (defaults to true if not specified)
- Supports updating from YouTube library view (no local file path required)

**Code Example:**
```javascript
const response = await youtube.videos.update({
  part: ['snippet', 'status'],
  requestBody: {
    id: videoId,
    snippet: {
      title: title.slice(0, 100),
      description: description,
      categoryId: categoryId,
      tags: tags || existingTags
    },
    status: {
      privacyStatus: privacy,
      embeddable: embeddable !== undefined ? embeddable : true
    }
  }
});
```

#### 3.3.3 Video Deletion (`videos.delete`)

**Implementation Location:** `src/main/youtube.js` - `deleteVideo()` method

**Process:**
1. Authenticates user and retrieves YouTube API client
2. Retrieves local video record to get YouTube video ID
3. Deletes video from YouTube using video ID
4. Removes local video record
5. Invalidates library cache

**Key Features:**
- Requires confirmation before deletion
- Removes both YouTube video and local record
- Error handling with clear messages

**Code Example:**
```javascript
await youtube.videos.delete({ id: videoId });
```

#### 3.3.4 Library Retrieval

**Implementation Location:** `src/main/youtube.js` - `getLibrarySnapshot()` method

**Process:**
1. Checks in-memory cache (60-second TTL by default)
2. If cache expired, checks disk cache
3. If no valid cache, fetches from YouTube API:
   - Retrieves channel information to get uploads playlist ID
   - Fetches up to 200 videos from uploads playlist
   - Retrieves detailed video information for each video
   - Normalizes and structures data
4. Caches results in memory and on disk
5. Returns structured library snapshot

**Caching Strategy:**
- **In-memory cache:** 60 seconds TTL (configurable via `DEFAULT_LIBRARY_CACHE_MS`)
- **Disk cache:** Persists between application sessions
- **Channel info cache:** 5 minutes TTL
- **Cache invalidation:** Triggered on upload, update, or delete operations

**Data Structure:**
```javascript
{
  accountId: string,
  channelId: string,
  channelTitle: string,
  fetchedAt: ISO8601 timestamp,
  videos: [
    {
      videoId: string,
      title: string,
      description: string,
      thumbnails: object,
      thumbnailUrl: string,
      publishedAt: ISO8601 timestamp,
      channelTitle: string,
      channelId: string,
      durationSeconds: number,
      liveBroadcastContent: 'none' | 'live' | 'upcoming',
      privacy: 'public' | 'private' | 'unlisted',
      embeddable: boolean,
      status: 'uploaded',
      type: 'videos' | 'shorts' | 'live',
      url: string,
      playlistIds: string[],
      accountId: string
    }
  ],
  playlists: [],
  playlistItems: {}
}
```

**Code Example:**
```javascript
// Fetch channel info
const { uploadsPlaylistId, channelId, channelTitle } = 
  await this.getChannelIdentifiers(youtube, accountId);

// Fetch playlist items
const uploadsItems = await this.fetchPlaylistItems(
  youtube, 
  uploadsPlaylistId, 
  200 // max videos
);

// Fetch detailed video info
const videos = await this.fetchVideosByIds(youtube, videoIds);
```

---

## 4. Data Handling and Privacy

### 4.1 Data Storage

All YouTube API data is stored **locally on the user's device**:

**Storage Locations:**
- `{userData}/youtube-videos-{accountId}.json` - Video records linking local files to YouTube videos
- `{userData}/youtube-cache/{accountId}.json` - Cached library data
- `{userData}/youtube-accounts.json` - OAuth account information (managed by OAuth module)

**Data Retention:**
- Video records: Persisted until user deletes them or removes account
- Library cache: Refreshed on-demand or when cache expires (default 60 seconds)
- OAuth tokens: Stored securely until user revokes access

**Data Scope:**
- Each authenticated user's data stored separately per account ID
- No data sharing between users
- No data transmission to external servers except YouTube API

### 4.2 Data Usage

**YouTube Data Usage:**
- Video metadata displayed in application UI
- Video thumbnails displayed in library view
- Channel information displayed in settings
- No derived metrics or analytics created from YouTube data
- No data exported or shared with third parties

**Local Data Usage:**
- Video records used to track upload status
- Library cache used to optimize API usage
- Account information used for authentication

### 4.3 Privacy Considerations

- **User Consent:** Users explicitly authenticate with Google to grant access
- **Scope Limitation:** Application requests only necessary scopes (youtube, youtube.upload, youtube.force-ssl)
- **Local Storage:** All data remains on user's device
- **No Third-Party Sharing:** No data shared with parties other than YouTube API
- **Account Isolation:** Each user's data stored separately
- **Transparent Operations:** All API operations logged for debugging (development mode only)

---

## 5. API Usage Optimization

### 5.1 Caching Strategy

The application implements a multi-tier caching strategy to minimize API calls:

1. **In-Memory Cache:**
   - Channel information: 5 minutes TTL
   - Library data: 60 seconds TTL (configurable)
   - Reduces redundant API calls within same session

2. **Disk Cache:**
   - Library data persisted between application sessions
   - Loaded on application start to avoid API calls
   - Updated when user explicitly refreshes or when cache expires

3. **Cache Invalidation:**
   - Triggered automatically on upload, update, or delete operations
   - Manual refresh option available to users
   - Ensures data consistency

### 5.2 API Call Optimization

- **Batched Requests:** Videos fetched in batches (50 per request)
- **Selective Updates:** Only requested parts updated (`snippet`, `status`)
- **Lazy Loading:** Library data loaded only when needed
- **Conditional Requests:** Cache checked before making API calls
- **Error Handling:** Graceful degradation when API calls fail

### 5.3 Quota Management

- **Caching:** Primary mechanism for quota conservation
- **User Education:** Clear error messages when quota limits approached
- **Efficient Operations:** Only necessary API calls made
- **Future Considerations:** Quota monitoring could be added to warn users before limits

---

## 6. Error Handling

### 6.1 Authentication Errors

- **Token Expiration:** Automatic token refresh handled by OAuth module
- **Revoked Access:** Clear error messages prompting re-authentication
- **Network Errors:** Retry logic with exponential backoff (handled by googleapis library)

### 6.2 API Errors

- **Quota Exceeded:** User-friendly error messages
- **Invalid Requests:** Validation before API calls
- **File Errors:** Validation of file existence and readability
- **Upload Errors:** Detailed error messages with suggestions

### 6.3 User Feedback

- Progress indicators for long-running operations
- Error banners with actionable messages
- Success confirmations for completed operations

---

## 7. Security Measures

### 7.1 OAuth Security

See Section 3.2.1 for detailed OAuth security implementation.

**Summary:**
- PKCE implementation for secure token exchange
- CSRF protection via state tokens
- Ephemeral session secrets in memory only
- Local redirect binding to 127.0.0.1 only

### 7.2 Data Security

- **Local Storage:** All data stored locally on user's device
- **Secure Token Storage:** OAuth tokens stored using Electron's secure storage
- **No Credential Sharing:** No credentials stored or transmitted insecurely
- **Input Validation:** All user inputs validated before API calls

### 7.3 API Security

- **HTTPS Only:** All API calls made over HTTPS
- **Scope Limitation:** Minimal required scopes requested
- **Token Refresh:** Automatic token refresh without user intervention
- **Account Isolation:** Each account's data handled separately

---

## 8. User Experience

### 8.1 Features

1. **Video Upload:**
   - Drag-and-drop or file picker for video selection
   - Custom metadata entry before upload
   - Real-time upload progress
   - Visual indicators for uploaded videos

2. **Library Management:**
   - Browse YouTube video library
   - View video thumbnails and metadata
   - Filter and search capabilities

3. **Metadata Management:**
   - Edit title, description, privacy settings
   - Update multiple videos in batch
   - Preview changes before applying

4. **Account Management:**
   - Support for multiple YouTube accounts
   - Easy account switching
   - Clear authentication status

### 8.2 UI Integration

- **Visual Indicators:**
  - Badges showing upload status
  - Progress bars for active uploads
  - Status icons for YouTube integration

- **Settings Panel:**
  - YouTube integration settings
  - Account connection/disconnection
  - Library refresh controls

---

## 9. Testing and Validation

### 9.1 Manual Testing Checklist

- ✓ Connect and disconnect YouTube account
- ✓ Upload video with custom metadata
- ✓ Verify upload progress tracking
- ✓ Update video metadata and verify changes on YouTube
- ✓ Delete video from YouTube and verify removal
- ✓ Test error handling (e.g., revoked credentials)
- ✓ Verify caching behavior
- ✓ Test multi-account support

### 9.2 Test Coverage

- Authentication flow
- Upload functionality
- Metadata updates
- Video deletion
- Library retrieval
- Cache behavior
- Error scenarios

---

## 10. Compliance with YouTube API Policies

### 10.1 Terms of Service Compliance

- ✓ Users authenticate with their own Google accounts
- ✓ Application requests only necessary scopes
- ✓ No unauthorized data access
- ✓ No sharing of user data with third parties
- ✓ Respects YouTube content policies
- ✓ No commercial use of YouTube Data
- ✓ Proper attribution where applicable

### 10.2 Developer Policies Compliance

- ✓ Secure OAuth implementation (PKCE)
- ✓ Proper error handling
- ✓ User privacy respected
- ✓ Transparent operations
- ✓ No misleading representations

---

## 11. Future Enhancements

Potential improvements (not currently implemented):

- Quota monitoring and user warnings
- Batch upload support
- Scheduled uploads
- Enhanced analytics integration
- Playlist management features
- Video category suggestions
- Thumbnail selection during upload

---

## 12. Technical Specifications

### 12.1 Dependencies

```json
{
  "googleapis": "^131.0.0",
  "electron": "^28.0.0"
}
```

### 12.2 Configuration

**OAuth Configuration File:** `config/google-oauth.json`
```json
{
  "installed": {
    "client_id": "...",
    "project_id": "miba-video-manager",
    "auth_uri": "https://accounts.google.com/o/oauth2/auth",
    "token_uri": "https://oauth2.googleapis.com/token",
    "client_secret": "...",
    "redirect_uris": ["http://localhost"]
  }
}
```

### 12.3 Constants

- `PLAYLIST_PAGE_SIZE = 50` - Videos fetched per playlist request
- `VIDEO_PAGE_SIZE = 50` - Videos fetched per videos.list request
- `DEFAULT_LIBRARY_CACHE_MS = 60 * 1000` - Default cache TTL (60 seconds)
- `CHANNEL_CACHE_MAX_AGE = 5 * 60 * 1000` - Channel info cache TTL (5 minutes)

---

## 13. Code References

Key implementation files:

- **Main YouTube Service:** `src/main/youtube.js`
- **OAuth Implementation:** `src/main/auth/googleOAuth.js`
- **Account Store:** `src/main/auth/youtubeAccountStore.js`
- **OAuth Documentation:** `docs/auth/google-oauth.md`
- **Main Entry Point:** `main.js`

---

## 14. Support and Contact

**Developer:** Michael Bachmann  
**Email:** miba@miba.dk  
**Website:** https://www.miba.dk/miba-video-manager  
**GitHub:** https://github.com/MiBaDK/video-manager

---

## 15. Conclusion

MiBa Video Manager implements YouTube Data API v3 in a secure, efficient, and user-friendly manner. The application follows Google's security best practices, implements proper caching to optimize API usage, and provides a seamless experience for content creators managing their YouTube content.

All YouTube API operations are performed with user consent through secure OAuth authentication, and all data is stored locally on the user's device with no external sharing. The application respects YouTube's Terms of Service and Developer Policies.

---

**Document Version:** 1.0  
**Last Updated:** 2025-01-XX  
**For:** YouTube API Services Quota Extension Request

